package ga.core;

import ga.core.individual.IIndividual;
import ga.core.individual.population.IPopulation;
import ga.core.validation.IValidator;

/**
 * Interface for a genetic algorithm. Stores also global constants.
 * 
 * @param <T>
 *          The generic type of individuals.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public interface GA<T extends IIndividual<T>> {
  public static final String KEY_VALIDATION_SPACE = "ValidationSpace";

  public static final String KEY_GENOME_MIN_LENGTH = "GenomeMinLength";
  public static final String KEY_GENOME_MAX_LENGTH = "GenomeMaxLength";

  public static final String KEY_INIT_INDIVIDUAL = "InitIndividual";
  public static final String KEY_INIT_INDIVIDUAL_PERCENTAGE = "InitIndividualPercentage";

  public static final String KEY_INTERVAL_FITNESS_MAX_WIDTH = "IntervalFitnessMaxWidth";

  public static final String KEY_EVALUATED_INDIVIDUALS_MIN_COUNT = "EvaluatedIndividualsMinCount";

  /**
   * Initializes the algorithm and population.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void init();

  /**
   * Do one step. This could create a new generation (generation-based GA) or do
   * one reproduction (steady state).
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void step();

  /**
   * Setter to enable/disable individual validation. To validate, this must be
   * set to <code>true</code> and a validator must be set.
   * 
   * @param validate
   *          Enable/disable validation.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void setValidate(final boolean validate);

  /**
   * Set the individual validator for the algorithm. To validate, this must be
   * not be <code>null</code> and validate must have been activated.
   * 
   * @param validator
   *          The individual validator.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void setValidator(final IValidator<T> validator);

  /**
   * Getter for the population.
   * 
   * @return The population.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public IPopulation<T> getPopulation();
}
